<?php
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

// Composer autoload (run `composer install` in project root)
if (file_exists(__DIR__ . '/../vendor/autoload.php')) {
    require_once __DIR__ . '/../vendor/autoload.php';
}

function get_setting($name) {
    global $conn;
    $stmt = $conn->prepare("SELECT option_value FROM settings WHERE option_name=?");
    $stmt->bind_param("s", $name);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($row = $result->fetch_assoc()) {
        return $row['option_value'];
    }
    return null;
}

function send_email($to, $subject, $message) {
    global $conn;
    $smtp_host = get_setting('smtp_host');
    $smtp_port = get_setting('smtp_port') ?: 587;
    $smtp_user = get_setting('smtp_user');
    $smtp_pass = get_setting('smtp_pass');
    $from_name = get_setting('email_from_name') ?: 'Visitor Gate Pass';
    $from_addr = get_setting('email_from_address') ?: 'noreply@example.com';

    // If PHPMailer is not installed, try PHP's mail() as a fallback.
    if (!class_exists('PHPMailer\\PHPMailer\\PHPMailer')) {
        $headers  = "MIME-Version: 1.0\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8\r\n";
        $headers .= "From: {$from_name} <{$from_addr}>\r\n";
        return @mail($to, $subject, $message, $headers);
    }

    $mail = new PHPMailer(true);
    try {
        $mail->isSMTP();
        $mail->Host = $smtp_host;
        $mail->SMTPAuth = true;
        $mail->Username = $smtp_user;
        $mail->Password = $smtp_pass;
        $mail->SMTPSecure = 'tls';
        $mail->Port = $smtp_port;
        $mail->setFrom($from_addr, $from_name);
        $mail->addAddress($to);
        $mail->isHTML(true);
        $mail->Subject = $subject;
        $mail->Body    = $message;
        $mail->send();
        return true;
    } catch (Exception $e) {
        error_log("Email Error: " . $mail->ErrorInfo);
        return false;
    }
}

function notify_host($visitor_name, $host_email, $purpose) {
    $subject = "Visitor Arrival Notification";
    $message = "
        <p>Dear Host,</p>
        <p><strong>{$visitor_name}</strong> has arrived.</p>
        <p>Purpose: {$purpose}</p>
        <p>Please attend to your visitor at reception.</p>
        <p>— Visitor Gate Pass</p>";
    send_email($host_email, $subject, $message);
}

function notify_visitor_checkout($visitor_email, $visitor_name) {
    if (!$visitor_email) return;
    $subject = "Thank You for Visiting";
    $message = "
        <p>Dear {$visitor_name},</p>
        <p>Thank you for visiting today.</p>
        <p>— Visitor Gate Pass</p>";
    send_email($visitor_email, $subject, $message);
}

function check_overstays() {
    global $conn;
    $limit_hours = (float) get_setting('overstay_limit_hours');
    $recipients = get_setting('overstay_recipients');

    if (!$limit_hours || !$recipients) return;
    $threshold_minutes = $limit_hours * 60;

    $sql = "
        SELECT v.id, v.name, v.email, v.checkin_time, h.name AS host_name
        FROM visitors v 
        LEFT JOIN hosts h ON v.host_id=h.id
        WHERE v.status='checked_in' 
          AND TIMESTAMPDIFF(MINUTE, v.checkin_time, NOW()) > {$threshold_minutes}
    ";
    $result = $conn->query($sql);

    if ($result->num_rows > 0) {
        while ($row = $result->fetch_assoc()) {
            $visitor_name = $row['name'];
            $host_name = $row['host_name'];
            $checkin = date('H:i', strtotime($row['checkin_time']));
            $subject = "Visitor Overstay Alert: {$visitor_name}";
            $message = "
                <p><strong>Overstay Alert</strong></p>
                <p>Visitor <strong>{$visitor_name}</strong> (Host: {$host_name}) 
                checked in at <strong>{$checkin}</strong> and has exceeded the configured stay limit.</p>
                <p>Please take appropriate action.</p>
                <p>— Visitor Gate Pass</p>";
            foreach (explode(',', $recipients) as $email) {
                $email = trim($email);
                if ($email) send_email($email, $subject, $message);
            }
        }
    }
}
?>
